<?php

namespace App\Http\Controllers;

use App\Models\Berita;
use App\Models\Produk;
use App\Models\Lowongan;
use App\Models\Demografi;
use App\Models\ProfilDesa;
use App\Models\Anggaran;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    public function index()
    {
        $latestBerita = Berita::where('status', 'published')
            ->orderBy('published_at', 'desc')
            ->take(4)
            ->get();
            
        $totalPenduduk = Demografi::where('kategori', 'Jenis Kelamin')
            ->sum('jumlah');
        
        // Total UMKM aktif untuk statistik di homepage
        $totalUmkmAktif = Produk::where('status', 'aktif')->count();
        
        // Profil aktif untuk sambutan kepala desa di beranda
        $profil = ProfilDesa::where('is_active', true)->first();
        
        // Anggaran total untuk transparansi di beranda
        $anggaranItems = Anggaran::orderBy('kategori')->get();
        $totalAnggaran = (float) Anggaran::sum('jumlah');
            
        return view('index', compact('latestBerita', 'totalPenduduk', 'totalUmkmAktif', 'profil', 'anggaranItems', 'totalAnggaran'));
    }
    
    public function tentang()
    {
        $demografiJenisKelamin = Demografi::where('kategori', 'Jenis Kelamin')->get();
        $demografiUsia = Demografi::where('kategori', 'Kelompok Usia')->get();
        $demografiPendidikan = Demografi::where('kategori', 'Pendidikan')->get();
        $demografiPekerjaan = Demografi::where('kategori', 'Pekerjaan')->get();
        
        return view('tentang', compact(
            'demografiJenisKelamin',
            'demografiUsia', 
            'demografiPendidikan',
            'demografiPekerjaan'
        ));
    }
    
    public function berita()
    {
        $berita = Berita::where('status', 'published')
            ->orderBy('published_at', 'desc')
            ->paginate(9);
            
        return view('berita.index', compact('berita'));
    }
    
    public function beritaDetail($slug)
    {
        $berita = Berita::where('slug', $slug)
            ->where('status', 'published')
            ->firstOrFail();
            
        $relatedBerita = Berita::where('status', 'published')
            ->where('id', '!=', $berita->id)
            ->orderBy('published_at', 'desc')
            ->take(3)
            ->get();
            
        return view('berita.detail', compact('berita', 'relatedBerita'));
    }
    
    public function produk()
    {
        $produk = Produk::where('status', 'aktif')
            ->orderBy('created_at', 'desc')
            ->paginate(12);
            
        $kategori = Produk::where('status', 'aktif')
            ->distinct()
            ->pluck('kategori');
            
        return view('produk.index', compact('produk', 'kategori'));
    }
    
    public function produkDetail($slug)
    {
        $produk = Produk::where('slug', $slug)
            ->where('status', 'aktif')
            ->firstOrFail();
            
        $relatedProduk = Produk::where('status', 'aktif')
            ->where('kategori', $produk->kategori)
            ->where('id', '!=', $produk->id)
            ->take(4)
            ->get();
            
        return view('produk.detail', compact('produk', 'relatedProduk'));
    }
    
    public function lowongan()
    {
        $lowongan = Lowongan::where('status', 'aktif')
            ->where('batas_lamaran', '>=', now())
            ->orderBy('created_at', 'desc')
            ->paginate(10);
            
        return view('lowongan.index', compact('lowongan'));
    }
    
    public function lowonganDetail($id)
    {
        $lowongan = Lowongan::where('id', $id)
            ->where('status', 'aktif')
            ->where('batas_lamaran', '>=', now())
            ->firstOrFail();
            
        return view('lowongan.detail', compact('lowongan'));
    }
    
    public function kontak()
    {
        return view('kontak');
    }
    
    public function demografi()
    {
        $demografiJenisKelamin = Demografi::where('kategori', 'Jenis Kelamin')->get();
        $demografiUsia = Demografi::where('kategori', 'Kelompok Usia')->get();
        $demografiPendidikan = Demografi::where('kategori', 'Pendidikan')->get();
        $demografiPekerjaan = Demografi::where('kategori', 'Pekerjaan')->get();
        $demografiAgama = Demografi::where('kategori', 'Agama')->get();
        
        // Get all categories for filter
        $tahunList = Demografi::distinct()->pluck('tahun')->sort()->values();
        
        return view('demografi', compact(
            'demografiJenisKelamin',
            'demografiUsia', 
            'demografiPendidikan',
            'demografiPekerjaan',
            'demografiAgama',
            'tahunList'
        ));
    }
    
    public function profil()
    {
        $profil = ProfilDesa::where('is_active', true)->first();
        
        // If no active profile exists, use default data or redirect
        if (!$profil) {
            // You can either show a default view or redirect to admin
            return view('profil', ['profil' => null]);
        }
        
        return view('profil', compact('profil'));
    }
}
